<?php
/**
 * utils
 * 
 * @package Asn
 * @author 	Fahd Saeed (fahds@fluidnets.com)
 */


/* ------------------------------- */
/* Core */
/* ------------------------------- */

/**
 * check_system_requirements
 * 
 * @return void
 */
function check_system_requirements()
{
    /* set required php version*/
    $required_php_version = '5.5';
    /* check php version */
    $php_version = phpversion();
    if (version_compare($required_php_version, $php_version, '>')) {
        _error("Installation Error", sprintf('<p class="text-center">Your server is running PHP version %1$s but INFER9X requires at least %3$s.</p>', $php_version, $required_php_version));
    }
    /* check if mysqli enabled */
    if (!extension_loaded('mysqli')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "mysqli" extension which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    /* check if curl enabled */
    if (!extension_loaded('curl')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "cURL" extension which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    /* check if intl enabled */
    if (!extension_loaded('intl')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "intl" extension which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    /* check if json_decode enabled */
    if (!function_exists('json_decode')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "json_decode()" function which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    /* check if base64_decode enabled */
    if (!function_exists('base64_decode')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "base64_decode()" function which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    /* check if mail enabled */
    if (!function_exists('mail')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "mail()" function which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
    if (!function_exists('getimagesize')) {
        _error("Installation Error", '<p class="text-center">Your PHP installation appears to be missing the "getimagesize()" function which is required by INFER9X.</p><small>Back to your server admin or hosting provider to enable it for you</small>');
    }
}


/**
 * get_system_protocol
 * 
 * @return string
 */
function get_system_protocol()
{
    $is_secure = false;
    if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on') {
        $is_secure = true;
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' || !empty($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] == 'on') {
        $is_secure = true;
    }
    return $is_secure ? 'https' : 'http';
}


/**
 * get_system_url
 * 
 * @return string
 */
function get_system_url()
{
    $protocol = get_system_protocol();
    $system_url = $protocol . "://" . $_SERVER['HTTP_HOST'] . BASEPATH;
    return rtrim($system_url, '/');
}

function log_to_console($data)
{
    $log_filename = ABSPATH . "/log.log";

    file_put_contents($log_filename, $data . "\n\n", FILE_APPEND);
}


/**
 * check_system_url
 * 
 * @return void
 */
function check_system_url()
{
    $protocol = get_system_protocol();
    $parsed_url = parse_url(SYS_URL);
    if (($parsed_url['scheme'] != $protocol) || ($parsed_url['host'] != $_SERVER['HTTP_HOST'])) {
        header('Location: ' . SYS_URL);
    }
}


/**
 * redirect
 * 
 * @param string $url
 * @return void
 */
function redirect($url = '/')
{
    if ($url) {
        header('Location: ' . SYS_URL . $url);
    } else {
        header('Location: ' . SYS_URL);
    }
    exit;
}

function filter_array($array, $fn)
{
    $filtered = [];

    foreach ($array as $item) {
        if ($fn($item))
            array_push($filtered, $item);
    }

    return $filtered;
}



/* ------------------------------- */
/* Security */
/* ------------------------------- */

/**
 * secure
 * 
 * @param string $value
 * @param string $type
 * @param boolean $quoted
 * @return string
 */
function secure($value, $type = "", $quoted = true)
{
    global $db;
    if ($value !== 'null') {
        // [1] Sanitize
        /* Escape all (single-quote, double quote, backslash, NULs) */
        if (get_magic_quotes_gpc()) {
            $value = stripslashes($value);
        }
        /* Convert all applicable characters to HTML entities */
        $value = htmlentities($value, ENT_QUOTES, 'utf-8');
        // [2] Safe SQL
        $value = $db->real_escape_string($value);
        switch ($type) {
            case 'int':
                $value = ($quoted) ? "'" . intval($value) . "'" : intval($value);
                break;
            case 'char':
                $value = ($quoted) ? "'" . $value . "'" : $value;
                break;
            case 'datetime':
                $value = ($quoted) ? "'" . set_datetime($value) . "'" : set_datetime($value);
                break;
            case 'search':
                if ($quoted) {
                    $value = (!is_empty($value)) ? "'%" . $value . "%'" : "''";
                } else {
                    $value = (!is_empty($value)) ? "'%%" . $value . "%%'" : "''";
                }
                break;
            default:
                $value = (!is_empty($value)) ? "'" . $value . "'" : "''";
                break;
        }
    }
    return $value;
}


/**
 * session_hash
 * 
 * @param string $hash
 * @return array
 */
function session_hash($hash)
{
}


/**
 * _password_hash
 * 
 * @param string $password
 * @return string
 */
function _password_hash($password)
{
    return password_hash($password, PASSWORD_DEFAULT);
}


/**
 * get_hash_key
 * 
 * @param integer $length
 * @return string
 */
function get_hash_key($length = 8)
{
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $count = mb_strlen($chars);
    for ($i = 0, $result = ''; $i < $length; $i++) {
        $index = mt_rand(0, $count - 1);
        $result .= mb_substr($chars, $index, 1);
    }
    return $result;
}


/**
 * get_hash_token
 * 
 * @return string
 */
function get_hash_token($len = 8)
{
    $rand = time() * mt_rand(1, 9999);
    return strtoupper(($len > 8) ? hash('fnv1a64', $rand) : hash('fnv1a32', $rand));

    /* $a =  hash_algos();
     $hashlen;
     foreach($a as $z => $s) {
         $hashlen[] = array($s => hash($a[$z], $rand;
     }
     return $hashlen;*/

}
function token_insert($token, $from, $expiry, $for = '', $value = '')
{
    global $db, $date;
    $db->query(sprintf("INSERT INTO `verification_token`(`token`, `for`, `from`, `value`, `expiry`, `date`) VALUES(%s, %s, %s, %s, %s, %s)", secure($token), secure($for), secure($from), secure($value), secure($expiry, 'int', false), secure($date))) or _error('SQL_ERROR_THROWEN');

    return $db->insert_id;
}

function token_info($id)
{
    global $db;
    $query = $db->query(sprintf("SELECT * FROM `verification_token` WHERE `id` = %s", secure($id))) or _error('SQL_ERROR_THROWEN');
    return $query->fetch_assoc();
}

function token_confirm($id, $token)
{
    global $db;
    $query = $db->query(sprintf("SELECT `token` FROM `verification_token` WHERE id = %s", secure($id))) or _error(SQL_ERROR_THROWEN);
    return ($token == $query->fetch_array()[0]) ? true : false;
}

function token_verify($id)
{
    global $db;
    $db->query(sprintf("UPDATE `verification_token` SET `verified` = '1' WHERE id = %s", secure($id))) or _error(SQL_ERROR_THROWEN);
}

function token_verified($id)
{
    global $db;
    $query = $db->query(sprintf("SELECT '1' FROM `verification_token` WHERE `id` = %s AND `verified` = '1'", secure($id))) or _error(SQL_ERROR_THROWEN);
    return (true == $query->fetch_array()[0]) ? true : false;
}

function token_delete($id)
{
    global $db;
    $db->query(sprintf("DELETE FROM verification_token WHERE id = %s", secure($id))) or _error(SQL_ERROR_THROWEN);
}

/* ------------------------------- */
/* Validation */
/* ------------------------------- */

/**
 * is_ajax
 * 
 * @return void
 */
function is_ajax()
{
    if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || ($_SERVER['HTTP_X_REQUESTED_WITH'] != 'XMLHttpRequest')) {
        redirect();
    }
}


/**
 * is_empty
 * 
 * @param string $value
 * @return boolean
 */
function is_empty($value)
{
    if (strlen(trim(preg_replace('/\xc2\xa0/', ' ', $value))) == 0) {
        return true;
    } else {
        return false;
    }
}


/**
 * valid_email
 * 
 * @param string $email
 * @return boolean
 */
function valid_email($email)
{
    if (filter_var($email, FILTER_VALIDATE_EMAIL) !== false) {
        return true;
    } else {
        return false;
    }
}

function valid_phone($phone)
{
    if (preg_match('/[+][0-9]{1,4}[0-9]{2}[0-9]{4,15}/', $phone)) {
        return true;
    } else {
        return false;
    }
}


/**
 * valid_url
 * 
 * @param string $url
 * @return boolean
 */
function valid_url($url)
{
    if (filter_var($url, FILTER_VALIDATE_URL, FILTER_FLAG_HOST_REQUIRED) !== false) {
        return true;
    } else {
        return false;
    }
}


/**
 * valid_username
 * 
 * @param string $username
 * @return boolean
 */
function valid_username($username)
{
    if (strlen($username) >= 3 && preg_match('/^[a-zA-Z0-9]+([_|.]?[a-zA-Z0-9])*$/', $username)) {
        return true;
    } else {
        return false;
    }
}


/**
 * reserved_username
 * 
 * @param string $username
 * @return boolean
 */
function reserved_username($username)
{
    $reserved_usernames = array('install', 'static', 'contact', 'contacts', 'sign', 'signin', 'login', 'signup', 'register', 'signout', 'logout', 'reset', 'activation', 'connect', 'revoke', 'packages', 'started', 'search', 'friends', 'messages', 'message', 'notifications', 'notification', 'settings', 'setting', 'posts', 'post', 'photos', 'photo', 'create', 'pages', 'page', 'groups', 'group', 'games', 'game', 'saved', 'directory', 'products', 'product', 'market', 'admincp', 'admin', 'admins', 'profile', 'channel', 'video', 'audio', 'sms', 'call', 'phone', 'messenger', 'forum', 'feeds');
    if (in_array(strtolower($username), $reserved_usernames)) {
        return true;
    } else {
        return false;
    }
}


/**
 * valid_name
 * 
 * @param string $name
 * @return boolean
 */
function valid_name($name)
{
    if (preg_match('/[\'^£$%&*()}{@#~?><>,|=+¬]/', $name)) {
        return false;
    } else {
        return true;
    }
}


/**
 * valid_location
 * 
 * @param string $location
 * @return boolean
 */
function valid_location($location)
{
    if (preg_match("/^[\\p{L} \-,()0-9]+$/ui", $location)) {
        return true;
    } else {
        return false;
    }
}


/**
 * valid_extension
 * 
 * @param string $extension
 * @param array $allowed_extensions
 * @return boolean
 */
function valid_extension($extension, $allowed_extensions)
{
    $extensions = explode(',', $allowed_extensions);
    foreach ($extensions as $key => $value) {
        $extensions[$key] = strtolower(trim($value));
    }
    if (is_array($extensions) && in_array($extension, $extensions)) {
        return true;
    }
    return false;
}



/* ------------------------------- */
/* Date */
/* ------------------------------- */

/**
 * set_datetime
 * 
 * @param string $date
 * @return string
 */
function set_datetime($date)
{
    return date("Y-m-d H:i:s", strtotime($date));
}


/**
 * get_datetime
 * 
 * @param string $date
 * @return string
 */
function get_datetime($date)
{
    return date("m/d/Y g:i A", strtotime($date));
}



/* ------------------------------- */
/* JSON */
/* ------------------------------- */

/**
 * _json_decode
 * 
 * @param string $string
 * @return string
 */
function _json_decode($string)
{
    if (get_magic_quotes_gpc())
        $string = stripslashes($string);
    return json_decode($string);
}

// function write_to_console($data) {
//  $console = $data;
//  if (is_array($console))
//  $console = implode(',', $console);

//  echo "<script>console.log('Console: " . $console . "' );</script>";
// }

/**
 * return_json
 * 
 * @param array $response
 * @return json
 */
function return_json($response = array())
{
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: POST, GET, DELETE, PUT, PATCH, OPTIONS');
        header('Access-Control-Allow-Headers: token, Content-Type');
        header('Access-Control-Max-Age: 1728000');
        header('Content-Length: 0');
        header('Content-Type: text/plain');
        die();
    }

    header('Access-Control-Allow-Origin: *');
    header('Content-Type: application/json');

    header('Expires: Sun, 01 Jan 2014 00:00:00 GMT');
    header('Cache-Control: no-store, no-cache, must-revalidate');
    header('Cache-Control: post-check=0, pre-check=0', FALSE);
    header('Pragma: no-cache');

    exit(json_encode($response));
}



/* ------------------------------- */
/* Error */
/* ------------------------------- */

/**
 * _error
 * 
 * @return void
 */
function _error()
{
    $args = func_get_args();
    if (count($args) > 1) {
        $title = $args[0];
        $message = $args[1];
    } else {
        switch ($args[0]) {
            case 'DB_ERROR':
                $title = "Database Error";
                $message = "<div class='text-left'><h1>" . "Error establishing a database connection" . "</h1>
                            <p>" . "This either means that the username and password information in your config.php file is incorrect or we can't contact the database server at localhost. This could mean your host's database server is down." . "</p>
                            <ul>
                                <li>" . "Are you sure you have the correct username and password?" . "</li>
                                <li>" . "Are you sure that you have typed the correct hostname?" . "</li>
                                <li>" . "Are you sure that the database server is running?" . "</li>
                            </ul>
                            </div>";
                break;

            case 'SQL_ERROR':
                $title = __("Database Error");
                $message = __("An error occurred while writing to database. Please try again later");
                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    $message .= "<br><br><small>This error function was called from line $line in file $file</small>";
                }
                break;

            case 'SQL_ERROR_THROWEN':
                $message = __("An error occurred while writing to database. Please try again later");

                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    $message .= "<br><br><small>This error function was called from line $line in file $file</small>";

                }
                die($message);
                // throw new Exception($message);


                break;

            case '404':
                header('HTTP/1.0 404 Not Found');
                $title = __("404 Not Found");
                $message = __("The requested URL was not found on this server. That's all we know");
                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    $message .= "<br><br><small>This error function was called from line $line in file $file</small>";
                }
                break;

            case '400':
                header('HTTP/1.0 400 Bad Request');
                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    exit("This error function was called from line $line in file $file");
                }
                exit;

            case '403':
                header('HTTP/1.0 403 Access Denied');
                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    exit("This error function was called from line $line in file $file");
                }
                exit;

            default:
                $title = __("Error");
                $message = __("Some thing went wrong");
                if (DEBUGGING) {
                    $backtrace = debug_backtrace();
                    $line = $backtrace[0]['line'];
                    $file = $backtrace[0]['file'];
                    $message .= "<br><br>" . "<small>This error function was called from line $line in file $file</small>";
                }
                break;
        }
    }
    echo '<!DOCTYPE html>
            <html>
            <head>
                <meta charset="utf-8">
                <title>' . $title . '</title>
                <style type="text/css">
                    html {
                        background: #f1f1f1;
                    }
                    body {
                        color: #555;
                        font-family: "Open Sans", Arial,sans-serif;
                        margin: 0;
                        padding: 0;
                    }
                    .error-title {
                        background: #ce3426;
                        color: #fff;
                        text-align: center;
                        font-size: 34px;
                        font-weight: 100;
                        line-height: 50px;
                        padding: 60px 0;
                    }
                    .error-message {
                        margin: 1em auto;
                        padding: 1em 2em;
                        max-width: 600px;
                        font-size: 1em;
                        line-height: 1.8em;
                        text-align: center;
                    }
                    .error-message .code,
                    .error-message p {
                        margin-top: 0;
                        margin-bottom: 1.3em;
                    }
                    .error-message .code {
                        font-family: Consolas, Monaco, monospace;
                        background: rgba(0, 0, 0, 0.7);
                        padding: 10px;
                        color: rgba(255, 255, 255, 0.7);
                        word-break: break-all;
                        border-radius: 2px;
                    }
                    h1 {
                        font-size: 1.2em;
                    }
                    
                    ul li {
                        margin-bottom: 1em;
                        font-size: 0.9em;
                    }
                    a {
                        color: #ce3426;
                        text-decoration: none;
                    }
                    a:hover {
                        text-decoration: underline;
                    }
                    .button {
                        background: #f7f7f7;
                        border: 1px solid #cccccc;
                        color: #555;
                        display: inline-block;
                        text-decoration: none;
                        margin: 0;
                        padding: 5px 10px;
                        cursor: pointer;
                        -webkit-border-radius: 3px;
                        -webkit-appearance: none;
                        border-radius: 3px;
                        white-space: nowrap;
                        -webkit-box-sizing: border-box;
                        -moz-box-sizing:    border-box;
                        box-sizing:         border-box;

                        -webkit-box-shadow: inset 0 1px 0 #fff, 0 1px 0 rgba(0,0,0,.08);
                        box-shadow: inset 0 1px 0 #fff, 0 1px 0 rgba(0,0,0,.08);
                        vertical-align: top;
                    }

                    .button.button-large {
                        height: 29px;
                        line-height: 28px;
                        padding: 0 12px;
                    }

                    .button:hover,
                    .button:focus {
                        background: #fafafa;
                        border-color: #999;
                        color: #222;
                        text-decoration: none;
                    }

                    .button:focus  {
                        -webkit-box-shadow: 1px 1px 1px rgba(0,0,0,.2);
                        box-shadow: 1px 1px 1px rgba(0,0,0,.2);
                    }

                    .button:active {
                        background: #eee;
                        border-color: #999;
                        color: #333;
                        -webkit-box-shadow: inset 0 2px 5px -3px rgba( 0, 0, 0, 0.5 );
                        box-shadow: inset 0 2px 5px -3px rgba( 0, 0, 0, 0.5 );
                    }
                    .text-left {
                        text-align: left;
                    }
                    .text-center {
                        text-align: center;
                    }
                </style>
            </head>
            <body>
                <div class="error-title">' . $title . '</div>
                <div class="error-message">' . $message . '</div>
            </body>
            </html>';
    exit;
}



/* ------------------------------- */
/* Email */
/* ------------------------------- */

/**
 * _email
 * 
 * @param string $email
 * @param string $subject
 * @param string $body
 * @param boolean $is_html
 * @param boolean $only_smtp
 * @return boolean
 */
function _email($email, $subject, $body, $is_html = false, $only_smtp = false)
{
    global $system;
    /* set header */
    $header = "MIME-Version: 1.0\r\n";
    $header .= "Mailer: " . $system['system_title'] . "\r\n";
    if ($system['system_email']) {
        $headers = "From: " . $system['system_email'] . "\r\n";
        $headers .= "Reply-To: " . $system['system_email'] . "\r\n";
    }
    if ($is_html) {
        $header .= "Content-Type: text/html; charset=\"utf-8\"\r\n";
    } else {
        $header .= "Content-Type: text/plain; charset=\"utf-8\"\r\n";
    }
    /* send email */
    if ($system['email_smtp_enabled']) {
        /* SMTP */
        require_once(ABSPATH . 'includes/libs/PHPMailer/PHPMailer.php');
        require_once(ABSPATH . 'includes/libs/PHPMailer/SMTP.php');
        require_once(ABSPATH . 'includes/libs/PHPMailer/Exception.php');
        $mail = new PHPMailer\PHPMailer\PHPMailer;
        $mail->CharSet = "UTF-8";
        $mail->isSMTP();
        $mail->Host = $system['email_smtp_server'];
        $mail->SMTPAuth = ($system['email_smtp_authentication']) ? true : false;
        $mail->Username = $system['email_smtp_username'];
        $mail->Password = $system['email_smtp_password'];
        $mail->SMTPSecure = ($system['email_smtp_ssl']) ? 'ssl' : 'tls';
        $mail->Port = $system['email_smtp_port'];
        $setfrom = (is_empty($system['email_smtp_setfrom'])) ? $system['email_smtp_username'] : $system['email_smtp_setfrom'];
        $mail->setFrom($setfrom, $system['system_title']);
        $mail->addAddress($email);
        $mail->Subject = $subject;
        if ($is_html) {
            $mail->isHTML(true);
            $mail->AltBody = strip_tags($body);
        }
        $mail->Body = $body;
        if (!$mail->send()) {
            if ($only_smtp) {
                return false;
            }
            /* send using mail() */
            if (!mail($email, $subject, $body, $header)) {
                return false;
            }
        }
    } else {
        if ($only_smtp) {
            return false;
        }
        /* send using mail() */
        if (!mail($email, $subject, $body, $header)) {
            return false;
        }
    }
    return true;
}


/**
 * email_smtp_test
 * 
 * @return void
 */
function email_smtp_test()
{
    global $system;
    /* prepare test email */
    $subject = __("Test SMTP Connection on") . " " . $system['system_title'];
    $body = __("This is a test email");
    $body .= "\r\n\r" . $system['system_title'] . " " . __("Team");
    /* send email */
    if (!_email($system['system_email'], $subject, $body, false, true)) {
        throw new Exception(__("Test email could not be sent. Please check your settings"));
    }
}



/* ------------------------------- */
/* SMS */
/* ------------------------------- */

/**
 * sms_send
 * 
 * @param string $phone
 * @param string $message
 * @return boolean
 */
function sms_send($phone, $message)
{
    global $system;
    require_once(ABSPATH . 'includes/libs/Twilio/autoload.php');
    $client = new Twilio\Rest\Client($system['twilio_sid'], $system['twilio_token']);
    $message = $client->account->messages->create(
        $phone,
        array(
            'from' => $system['twilio_phone'],
            'body' => $message
        )
    );
    if (!$message->sid) {
        return false;
    }
    return true;
}


/**
 * sms_test
 * 
 * @return void
 */
function sms_test()
{
    global $system;
    require_once(ABSPATH . 'includes/libs/Twilio/autoload.php');
    $client = new Twilio\Rest\Client($system['twilio_sid'], $system['twilio_token']);
    $message = $client->account->messages->create(
        $system['system_phone'],
        array(
            'from' => $system['twilio_phone'],
            'body' => __("Test SMS from") . " " . $system['system_title']
        )
    );
    if (!$message->sid) {
        throw new Exception(__("Test SMS could not be sent. Please check your settings"));
    }
}


/* ------------------------------- */
/* User Access */
/* ------------------------------- */

/**
 * user_access
 * 
 * @param boolean $is_ajax
 * @return void
 */
function user_access($is_ajax = false, $subscribe = true)
{
    global $user, $system;
    if ($is_ajax) {
        /* check user logged in */
        if (!$user->_logged_in) {
            modal('LOGIN');
        }
        /* check user activated */
        // if($system['activation_enabled'] && !$user->_data['activated']) {
        //     modal(MESSAGE, __("Not Activated"), __("Before you can interact with other users, you need to confirm your email address"));
        //}
    } else {
        if (!$user->_logged_in) {
            user_login();

        } else {
            /* check if getting started */
            if ($system['getting_started'] && !$user->_data['started']) {
                redirect('/started');
            }
        }
    }
}


/**
 * user_login
 * 
 * @return void
 */
function user_login()
{
    global $smarty;
    $smarty->assign('highlight', __("You must sign in to see this page"));
    //page_info(__("Sign in"));
    page_display('sign');
    exit;
}



/* ------------------------------- */
/* Modal */
/* ------------------------------- */

/**
 * modal
 * 
 * @return json
 */
function modal()
{
    $args = func_get_args();
    switch ($args[0]) {
        case 'LOGIN':
            return_json(array("callback" => "modal('#modal-login')"));
            break;
        case 'MESSAGE':
            return_json(array("callback" => "modal('#modal-message', {title: '" . $args[1] . "', message: '" . addslashes($args[2]) . "'})"));
            break;
        case 'ERROR':
            return_json(array("callback" => "modal('#modal-error', {title: '" . $args[1] . "', message: '" . addslashes($args[2]) . "'})"));
            break;
        case 'SUCCESS':
            return_json(array("callback" => "modal('#modal-success', {title: '" . $args[1] . "', message: '" . addslashes($args[2]) . "'})"));
            break;
        default:
            if (isset($args[1])) {
                return_json(array("callback" => "modal('" . $args[0] . "', " . $args[1] . ")"));
            } else {
                return_json(array("callback" => "modal('" . $args[0] . "')"));
            }
            break;
    }
}



/* ------------------------------- */
/* Popover */
/* ------------------------------- */

/**
 * popover
 * 
 * @param integer $uid
 * @param string $username
 * @param string $name
 * @return string
 */
function popover($uid, $username, $name)
{
    global $system;
    $popover = '<span class="js_user-popover" data-uid="' . $uid . '"><a href="' . $system['system_url'] . '/' . $username . '">' . $name . '</a></span>';
    return $popover;
}



/* ------------------------------- */
/* Page */
/* ------------------------------- */
/**
 * page_info
 * 
 * @param string $title
 * @param string $description
 * @return void
 */
function page_info($title, $description = '', $image = '')
{
    global $smarty, $system;
    /**
       $description = ($description != '')? $description : $system['system_description'];
       if($image == '') {
           if($system['system_ogimage']) {
               $image = $system['system_uploads'].'/'.$system['system_ogimage'];
           } else {
               $image = $system['system_url'].'/content/themes/'.$system['theme'].'/images/og-image.jpg';
           }
       }*/
    $smarty->assign('page_title', $title);
    $smarty->assign('page_description', $description);
    $smarty->assign('page_image', $image);
}


/**
 * page_display
 * 
 * @param string $page
 * @return void
 */
function page_display($page)
{
    global $smarty;
    $smarty->assign('page', $page);
    $smarty->display("$page.tpl");
}



/* ------------------------------- */
/* Text */
/* ------------------------------- */

/**
 * decode_urls
 * 
 * @param string $text
 * @return string
 */
function decode_urls($text)
{
    $text = preg_replace('/(https?:\/\/[^\s]+)/', "<a target='_blank' href=\"$1\">$1</a>", $text);
    return $text;
}


/**
 * decode_text
 * 
 * @param string $string
 * @return string
 */
function decode_text($string)
{
    return base64_decode($string);
}



/* ------------------------------- */
/* Censored Words */
/* ------------------------------- */

/**
 * censored_words
 * 
 * @param string $text
 * @return string
 */
function censored_words($text)
{
    global $system;
    if ($system['censored_words_enabled']) {
        $bad_words = explode(',', $system['censored_words']);
        if (count($bad_words) > 0) {
            foreach ($bad_words as $word) {
                $pattern = '/' . $word . '/i';
                $text = preg_replace($pattern, str_repeat('*', strlen($word)), $text);
            }
        }
    }
    return $text;
}



/* ------------------------------- */
/* General */
/* ------------------------------- */

/**
 * get_ip
 * 
 * @return string
 */
function get_user_ip()
{
    /* handle CloudFlare IP addresses */
    return (isset($_SERVER["HTTP_CF_CONNECTING_IP"]) ? $_SERVER["HTTP_CF_CONNECTING_IP"] : $_SERVER['REMOTE_ADDR']);
}


/**
 * get_os
 * 
 * @return array [platform|category]
 */
function get_user_os()
{
    $os_platform = "Unknown OS Platform";
    $os_category = "Unknown OS Category";
    $os_array = array(
        '/windows nt 10/i' => array('Windows 10', 'desktop'),
        '/windows nt 6.3/i' => array('Windows 8.1', 'desktop'),
        '/windows nt 6.2/i' => array('Windows 8', 'desktop'),
        '/windows nt 6.1/i' => array('Windows 7', 'desktop'),
        '/windows nt 6.0/i' => array('Windows Vista', 'desktop'),
        '/windows nt 5.2/i' => array('Windows Server 2003/XP x64', 'desktop'),
        '/windows nt 5.1/i' => array('Windows XP', 'desktop'),
        '/windows xp/i' => array('Windows XP', 'desktop'),
        '/windows nt 5.0/i' => array('Windows 2000', 'desktop'),
        '/windows me/i' => array('Windows ME', 'desktop'),
        '/win98/i' => array('Windows 98', 'desktop'),
        '/win95/i' => array('Windows 95', 'desktop'),
        '/win16/i' => array('Windows 3.11', 'desktop'),
        '/macintosh|mac os x/i' => array('Mac OS X', 'desktop'),
        '/mac_powerpc/i' => array('Mac OS 9', 'desktop'),
        '/linux/i' => array('Linux', 'desktop'),
        '/ubuntu/i' => array('Ubuntu', 'desktop'),
        '/iphone/i' => array('iPhone', 'desktop'),
        '/ipod/i' => array('iPod', 'mobile'),
        '/ipad/i' => array('iPad', 'mobile'),
        '/android/i' => array('Android', 'mobile'),
        '/blackberry/i' => array('BlackBerry', 'mobile'),
        '/webos/i' => array('Mobile', 'mobile'),
    );
    foreach ($os_array as $regex => $value) {
        if (preg_match($regex, $_SERVER['HTTP_USER_AGENT'])) {
            $os_platform = $value[0];
            $os_category = $value[1];
        }
    }
    return array('platform' => $os_platform, 'category' => $os_category);
}


/**
 * get_browser
 * 
 * @return string
 */
function get_user_browser()
{
    $browser = "Unknown Browser";
    $browser_array = array(
        '/msie/i' => 'Internet Explorer',
        '/firefox/i' => 'Firefox',
        '/safari/i' => 'Safari',
        '/chrome/i' => 'Chrome',
        '/edge/i' => 'Edge',
        '/opera/i' => 'Opera',
        '/netscape/i' => 'Netscape',
        '/maxthon/i' => 'Maxthon',
        '/konqueror/i' => 'Konqueror',
        '/mobile/i' => 'Handheld Browser'
    );
    foreach ($browser_array as $regex => $value) {
        if (preg_match($regex, $_SERVER['HTTP_USER_AGENT'])) {
            $browser = $value;
        }
    }
    return $browser;
}


/**
 * get_extension
 * 
 * @param string $path
 * @return string
 */
function get_extension($path)
{
    return strtolower(pathinfo($path, PATHINFO_EXTENSION));
}


/**
 * ger_origenal_url
 * 
 * @param string $url
 * @return string
 */
function get_original_url($url)
{
    stream_context_set_default(
        array(
            'http' => array(
                'ignore_errors' => true,
                'method' => 'HEAD',
                'user_agent' => @$_SERVER['HTTP_USER_AGENT'],
            )
        )
    );
    $headers = get_headers($url, 1);
    if ($headers !== false && (isset($headers['location']) || isset($headers['Location']))) {
        $location = (isset($headers['location'])) ? $headers['location'] : $headers['Location'];
        return is_array($location) ? array_pop($location) : $location;
    }
    return $url;
}


/**
 * get_url_text
 * 
 * @param string $string
 * @param integer $length
 * @return string
 */
function get_url_text($string, $length = 10)
{
    $string = htmlspecialchars_decode($string, ENT_QUOTES);
    $string = preg_replace('/[^\\pL\d]+/u', '-', $string);
    $string = trim($string, '-');
    $words = explode("-", $string);
    if (count($words) > $length) {
        $string = "";
        for ($i = 0; $i < $length; $i++) {
            $string .= "-" . $words[$i];
        }
        $string = trim($string, '-');
    }
    return $string;
}


/**
 * remove_querystring_var
 * 
 * @param string $url
 * @param string $key
 * @return string
 */
function remove_querystring_var($url, $key)
{
    $url = preg_replace('/(.*)(?|&)' . $key . '=[^&]+?(&)(.*)/i', '$1$2$4', $url . '&');
    $url = substr($url, 0, -1);
    return $url;
}


/**
 * get_snippet_text
 * 
 * @param string $string
 * @return string
 */
function get_snippet_text($string)
{
    $string = htmlspecialchars_decode($string, ENT_QUOTES);
    $string = strip_tags($string);
    return $string;
}


/**
 * get_tag
 * 
 * @param string $string
 * @return string
 */
function get_tag($string)
{
    $string = trim($string);
    $string = preg_replace('/[^\\pL\d]+/u', '_', $string);
    return $string;
}


/**
 * get_youtube_id
 * 
 * @param string $url
 * @return string
 */
function get_youtube_id($url)
{
    preg_match('/youtube\.com\/embed\/([^\&\?\/]+)/', $url, $id);
    return $id[1];
}


/**
 * get_array_key
 * 
 * @param array $array
 * @param integer $current
 * @param integer $offset
 * @return mixed
 */
function get_array_key($array, $current, $offset = 1)
{
    $keys = array_keys($array);
    $index = array_search($current, $keys);
    if (isset($keys[$index + $offset])) {
        return $keys[$index + $offset];
    }
    return false;
}

?>